import React, { useEffect, useState } from "react";
import { View, Text, TouchableOpacity, FlatList } from "react-native";
import { useDispatch, useSelector } from "react-redux";
import useAuth from "../../hooks/useAuth";
import { ProfileStyles } from "./ProfileStyle";
import MenuNavigation from "../Utilities/CustomHeader/MenuNavigation";
import EditProfileIcon from "../../assets/svgs/profile/edit profile.svg";
import WishIcon from "../../assets/icons/icon-wish.svg";
import BagIcon from "../../assets/icons/icon-bag.svg";
import ListIcon from "../../assets/icons/icon-list.svg";
import ProfileSkeleton from "../../src/skeletons/screens/profile/ProfileSkeleton";
import { login } from "../../redux/slices/auth/signIn";
import rootApiRequest, { authApiRequest } from "../../helper/rootApiRequest";
import { queryResponse } from "../../redux/slices/util/processResponse.js";
import { InOnScreenLoader } from "../ShoppingCart/ShoppingCart";
import { updateUserProfile } from "../../redux/slices/user/updateProfile/getUpdateProfile";
import resetReduxStore from "../../helper/resetReduxStore";
import { getPreferences } from "../../redux/slices/preferences/getPreferences";
import { BASE_API_URL_STG } from "@env";
import ProfilePicture from "./ProfilePicture";
import dpr from "../Utilities/CustomStyleAttribute/dpr";
import useLangTranslation from "../../hooks/useLangTranslation";
import gapStyle from "../Utilities/CommonStyles/gap.style";
import CustomSVG from "../../src/components/CustomSVG/CustomSVG";

const Profile = (props) => {
    const dispatch = useDispatch();
    const { trans } = useLangTranslation();
    const { access_token, user } = useAuth();
    const { loading } = useSelector((state) => state.updateUserProfile);
    const { pagination: wishlistPagination } = useSelector((state) => state.getMyWishlistSlice);
    const { pagination } = useSelector((state) => state.getMyOrdersSlice);

    const {
        cart_Data: { cartData },
    } = useSelector((state) => state.cartProductSlice);
    const [logoutLoading, setLogoutLoading] = useState(false);
    

    useEffect(() => {
        handleRefresh();
    }, []);

    const handleLogout = async () => {
        const LOGOUT_URL = `${BASE_API_URL_STG}/user/logout`;
        setLogoutLoading(true);
        const response = await queryResponse(LOGOUT_URL, "GET", access_token);
        setLogoutLoading(false);
        resetReduxStore(dispatch);
        rootApiRequest(dispatch, access_token);
    };

    const handleRefresh = async () => {
        dispatch(getPreferences());
        try{
            const data = await dispatch(updateUserProfile(access_token));
            if(data?.payload?.status?.code == 200) {
                const payload = {
                    access_token: data?.meta?.arg,
                    user: data?.payload?.records?.data,
                }
                dispatch(login(payload))
            }
        } catch (err){}
        authApiRequest(dispatch, access_token);
    };

    return (
        <View style={ProfileStyles.wrapper}>
            {logoutLoading && <InOnScreenLoader />}
            <MenuNavigation
                navigationProps={props.navigation}
                routeName={trans("MY ACCOUNT")}
            />
            <View style={ProfileStyles.hrLine} />
            <FlatList
                showsVerticalScrollIndicator={false}
                ListHeaderComponent={() => (
                    <View style={[
                        ProfileStyles.profileContainer,
                        ProfileStyles.profileSecondContainer
                    ]}>
                        {loading ? (
                            <ProfileSkeleton />
                        ) : (
                            <View style={ProfileStyles.profileImageCont}>
                                <ProfilePicture user={user} />
                                <View style={ProfileStyles.profileDesc}>
                                    <View>
                                        <Text style={ProfileStyles.name}>
                                            {user?.name}
                                        </Text>
                                        <Text style={ProfileStyles.email}>
                                            {user?.email}
                                        </Text>
                                    </View>
                                    <TouchableOpacity
                                        onPress={() =>
                                            props.navigation.navigate(
                                                "edit profile"
                                            )
                                        }
                                    >
                                        <View style={ProfileStyles.editProfile}>
                                            <CustomSVG svgIcon={EditProfileIcon}
                                                width={dpr(16)}
                                                height={dpr(24)}
                                            />
                                            <Text
                                                style={ProfileStyles.editText}
                                            >
                                                {trans("Edit profile")}
                                            </Text>
                                        </View>
                                    </TouchableOpacity>
                                </View>
                            </View>
                        )}
                        <View style={ProfileStyles.profileInfoContainer}>
                            <ProfileInfoCard
                                bgcolor="#FCCA19"
                                iconComponent={<CustomSVG svgIcon={BagIcon} />}
                                headerContent={cartData?.length || 0}
                                bodyContent={trans("In Cart")}
                                onPress={() => props.navigation.navigate("my cart")}
                            />
                            <ProfileInfoCard
                                bgcolor="#2C2C2C"
                                iconComponent={<CustomSVG svgIcon={WishIcon} />}
                                headerContent={wishlistPagination?.total || 0}
                                bodyContent={trans("Wishlist")}
                                onPress={() => props.navigation.navigate("my wishlist")}
                            />
                            <ProfileInfoCard
                                bgcolor="#2C2C2C"
                                iconComponent={<CustomSVG svgIcon={ListIcon} />}
                                headerContent={pagination?.total || 0}
                                bodyContent={trans("All Orders")}
                                onPress={() => props.navigation.navigate("order history")}
                            />
                        </View>
                        <View>
                            <View
                                style={[
                                    ProfileStyles.infoBorder,
                                    gapStyle.pt24,
                                ]}
                            >
                                <TouchableOpacity
                                    onPress={() => {
                                        props.navigation.navigate(
                                            "address book"
                                        );
                                    }}
                                >
                                    <Text style={ProfileStyles.info}>
                                        {trans("My Address")}
                                    </Text>
                                </TouchableOpacity>
                            </View>
                            <View style={ProfileStyles.infoBorder}>
                                <TouchableOpacity
                                    onPress={() => {
                                        props.navigation.navigate(
                                            "order history"
                                        );
                                    }}
                                >
                                    <Text style={ProfileStyles.info}>
                                        {trans("Order History")}
                                    </Text>
                                </TouchableOpacity>
                            </View>
                            <View style={ProfileStyles.infoBorder}>
                                <TouchableOpacity
                                    onPress={() =>
                                        props.navigation.navigate("my reviews")
                                    }
                                >
                                    <Text style={ProfileStyles.info}>
                                        {trans("My Reviews")}
                                    </Text>
                                </TouchableOpacity>
                            </View>
                            <View style={ProfileStyles.infoBorder}>
                                <TouchableOpacity
                                    onPress={() =>
                                        props.navigation.navigate("my refund")
                                    }
                                >
                                    <Text style={ProfileStyles.info}>
                                        {trans("Refund Requests")}
                                    </Text>
                                </TouchableOpacity>
                            </View>
                            <View style={ProfileStyles.infoBorder}>
                                <TouchableOpacity
                                    onPress={() =>
                                        props.navigation.navigate("settings")
                                    }
                                >
                                    <Text style={ProfileStyles.info}>
                                        {trans("Settings")}
                                    </Text>
                                </TouchableOpacity>
                            </View>
                            <TouchableOpacity onPress={handleLogout}>
                                <Text
                                    style={[
                                        ProfileStyles.info,
                                        gapStyle.pb0
                                    ]}
                                >
                                    {trans("Logout")}
                                </Text>
                            </TouchableOpacity>
                        </View>
                    </View>
                )}
                onRefresh={handleRefresh}
                refreshing={false}
            />
        </View>
    );
};

export default Profile;

const ProfileInfoCard = ({ bgcolor, iconComponent, headerContent, bodyContent, onPress=() => {} }) => {
    const theme = bgcolor === "#FCCA19" ? "light" : "dark";
    return (
        <TouchableOpacity onPress={onPress} activeOpacity={0.8} style={ProfileStyles.profileInfoCard(bgcolor)}>
            <View style={ProfileStyles.profileInfoCardHeader}>
                {iconComponent}
                <Text style={ProfileStyles.profileInfoCardHeaderText(theme)}>{headerContent || 0}</Text>
            </View>
            <Text style={ProfileStyles.profileInfoCardBody(theme)}>{bodyContent}</Text>
        </TouchableOpacity>
    )
}
